
####################################
####  pathway scoring analysis  ####

#### four functions: a wrapper, one to fit scores from a prior .csv, one to fit PC1, one to plot results:

### wrapper function:
# data = normalized data matrix (no distinguishing between analyze types)
# gene.sets.list: named list of gene set vectors for PC1 scoring
# adjust.for: data frame of covariates to adjust for. ONLY used under PC1 <--------------------
# signaturematrix: matrix of gene weights for predefined signatures
# requireallgenes: T/F for whether to use incomplete gene signatures (ie those with some genes not in codeset)
# mingenes: min number of genes for a geneset or signature to be calculated
# prunedprobes: only removed from PC1 scoring
run.pathway.scoring = function(data,path.scoring.method,
                               prb.set.matrix=NULL,mingenes.pc1=5,prunedprobes=NULL,adjust.for=NULL, #PC1 arguments
                               signaturematrix.predefined=NULL,requireallgenes=FALSE,mingenes.existing=5, # existing scores arguments
                               plot.against.covariates,covariates,covariate.type,annotcols,annotcols2,
                               log,plottypearg,path.results,path.inc,path.to.csvs,path.to.pathscore.results)
{
  
  ## first, fit the score using whichever method:
  if(path.scoring.method=="PC1")
  {
    #convert the prb.set.matrix to gene.sets.list:
    gene.sets.list = list()
    for(i in 1:ncol(prb.set.matrix))
    {
      #print(length(rownames(prb.set.matrix)[(prb.set.matrix[,i]!=0)&!is.na((prb.set.matrix[,i]))]))
      if(length(rownames(prb.set.matrix)[(prb.set.matrix[,i]!=0)&!is.na((prb.set.matrix[,i]))]) < 1)
      {
        gene.sets.list[[i]] = NA
        names(gene.sets.list)[i] = colnames(prb.set.matrix)[i]
      }
      else{
        gene.sets.list[[i]] = rownames(prb.set.matrix)[(prb.set.matrix[,i]!=0)&!is.na((prb.set.matrix[,i]))]
        names(gene.sets.list)[i] = colnames(prb.set.matrix)[i]
      }
    }

    #names(gene.sets.list)=colnames(prb.set.matrix)

    gene.sets.list        <- lapply(gene.sets.list, function (x) x[!is.na(x)]) 
    gene.sets.list.final  <- gene.sets.list[lapply(gene.sets.list, length) > 0]
    
    temp = calc.PC1.scores(data=data,gene.sets.list=gene.sets.list.final,
                           omitlowgenes=TRUE,prunedprobes=NULL,min.obs = 12,mingenes.pc1=mingenes.pc1)
    sigs = temp$sigs
    signaturematrix.final = temp$signaturematrix
    signaturematrix.predefined = NULL
  }
  if(path.scoring.method=="predefined")
  {
    temp = calculate.linear.signatures(data=data,signaturematrix=signaturematrix.predefined,requireallgenes=requireallgenes,mingenes=1)
    sigs = temp$sigs
    signaturematrix.final = temp$signaturematrix.present
  }
  ## draw the plots:
  temp = analyze.signatures(sigs,signaturematrix.predefined,signaturematrix.final,data,path.scoring.method,
                            plot.against.covariates,covariates,covariate.type,annotcols,annotcols2,
                            log,plottypearg,path.results,path.inc,path.to.csvs,path.to.pathscore.results)
    
}


#####

calc.PC1.scores = function(data,gene.sets.list,omitlowgenes=TRUE,prunedprobes=NULL,min.obs = 12, min.sd = 1, mingenes.pc1,adjust.for=NULL)
{
  # prune if needed:
  data = data[,!is.element(colnames(data),prunedprobes)]
  
  # adjust for covariates if indicated:
  if(length(adjust.for)>0)
  {
    # Check the degrees of freedom:
    temp = cbind(data[,1],adjust.for)
    if(summary(lm(temp))$df[2]<1)
    {
      warning("Warning: Too many parameters or levels to adjust for given the number of observations - Pathway scoring will not be run.") 
      return(scores=NULL)
    }
    adjusted=adjusted*NA
    for(i in 1:dim(adjusted)[2])
    {
      # combine with adjust.for to create a data frame on which to run a lm:
      temp = cbind(data[,i],adjust.for)
      mod = lm(temp)
      # fill in the "adjusted" data frame with the residuals of the lm:
      adjusted[names(mod$resid),i] = mod$resid
    }
    # remove NA rows:
    adjusted = adjusted[rowSums(is.na(adjusted))==0,]
    # replace the "data" object: 
    data = adjusted
  }
  
  # for each gene set, calculate PC1:
  sigs = matrix(NA,nrow(data),length(gene.sets.list)); dimnames(sigs)=list(rownames(data),names(gene.sets.list))
  signaturematrix = matrix(0,ncol(data),length(gene.sets.list)); dimnames(signaturematrix)=list(colnames(data),names(gene.sets.list))
  for(k in 1:length(gene.sets.list))
  {
    sharedgenes = intersect(colnames(data),gene.sets.list[[k]])
    if(length(sharedgenes)>=mingenes.pc1)
    {
      temp = data[,sharedgenes]
      # scale the data appropriately:
      scaling.factors = sqrt(apply(temp,2,var)+min.sd^2)
      temp2 = t(t(temp)/scaling.factors)
      # run PCA:
      pct = prcomp(temp2,scale.=FALSE)
      # reorient the signature if the average gene's weight is negative:
      sigsign = sign(mean(pct$rotation[,1])); if(sigsign==0){sigsign=1}
      # save output:
      sigs[,k] = pct$x[,1]*sigsign
      signaturematrix[sharedgenes,k] = pct$rotation[,1]*sigsign  ## <---------------- need to undo the scaling!!!!
    }
  }
  sigs <- sigs[,colSums(is.na(sigs)) < nrow(sigs)]
  #signaturematrix <- signaturematrix[,colSums(is.na(sigs)) < nrow(sigs)]
  signaturematrix <- signaturematrix[,which(colSums(signaturematrix) != 0)]
  out = list(sigs=sigs,signaturematrix=signaturematrix)
  return(out)
}


# data: the normalized data
# signaturematrix: a matrix of signatures, 
# requireallgenes: TRUE/FALSE: delete signatures with genes not present in the data
calculate.linear.signatures = function(data,signaturematrix,requireallgenes,mingenes=1)
{
  # separate the signaturematrix into genes we have and genes we don't:
  genes.in.codeset = intersect(colnames(data),rownames(signaturematrix))
  genes.not.in.codeset = setdiff(rownames(signaturematrix),colnames(data))
  signaturematrix.missing = signaturematrix[genes.not.in.codeset,,drop=FALSE]
  signaturematrix.present = signaturematrix[genes.in.codeset,,drop=FALSE]
  
  # screen out signatures with no genes.  if use.all.genes == FALSE, also screen out signatures with only 1 gene:
  signature.too.few.genes = colSums(!is.na(signaturematrix.present))<mingenes
  signaturematrix.present = signaturematrix.present[,!signature.too.few.genes,drop=FALSE]
  
  # if we require all genes to be present, drop signatures w lost genes:
  if(requireallgenes)
  {
    sigs.w.lost.genes = colSums(signaturematrix.missing!=0,na.rm=T)>0
    signaturematrix.present = signaturematrix.present[,sigs.w.lost.genes,drop=FALSE]
  }
  
  ## calculate signature scores:
  sigs = data[,genes.in.codeset]%*%signaturematrix.present[genes.in.codeset,]
  out = list(sigs=sigs,signaturematrix.present=signaturematrix.present)
  return(out)
}

## QC: PCA, and analysis of coverage of signatures (if requireallgenes=FALSE)
analyze.signatures = function(sigs,signaturematrix.predefined,signaturematrix.final,data,path.scoring.method,
                              plot.against.covariates,covariates,covariate.type,annotcols,annotcols2,
                              log,plottypearg,path.results,path.inc,path.to.csvs,path.to.pathscore.results)
{

  covariates <- covariates[ ,colnames(annotcols)[colnames(annotcols) %in% colnames(covariates)] ,drop=F]

  ## color scheme for colored PCA plots:
  pairscols = list(); pairscols[[1]]="black"
  covariate.names.for.pairsplots = c("")
  #if(plot.against.covariates&(dim(covariates)[2]>0))
  if(plot.against.covariates&(length(covariates)>0))
  {
    for(i in 1:dim(covariates)[2])
    {
      pairscols[[i]] = annotcols[,colnames(covariates)[i]]   #annotcols[,i]
      covariate.names.for.pairsplots[i] = dimnames(covariates)[[2]][i]
    }
  }
  # parse the signature matrices:

  if(is.null(signaturematrix.predefined) == F) signaturematrix.predefined = replace(signaturematrix.predefined,is.na(signaturematrix.predefined),0)

  #### QC plots for only the predefined method:  
  if(path.scoring.method=="predefined")
  {
    ## if we're not requiring all genes, then summarize the panel's coverage of each signature. Summarize which signatures were dropped.
    signatures.all = colnames(signaturematrix.predefined)
    ngenes.present = colSums(((signaturematrix.predefined!=0)&!is.na(signaturematrix.predefined)) * is.element(rownames(signaturematrix.predefined),colnames(data)))
    ngenes.missing = colSums(((signaturematrix.predefined!=0)&!is.na(signaturematrix.predefined)) * !is.element(rownames(signaturematrix.predefined),colnames(data)))
    signatures.failed = !is.element(signatures.all,colnames(sigs))
    missing.genenames = rep("",length(signatures.all))
    names(missing.genenames)=signatures.all
    for(i in 1:length(signatures.all))
    {
      tempsig = signaturematrix.predefined[,signatures.all[i]]
      temp.missing.genenames = setdiff(rownames(signaturematrix.predefined)[(tempsig!=0)&(!is.na(tempsig))],colnames(data))
      missing.genenames[i] = paste(temp.missing.genenames,collapse=" ")
    }
    sig.qc.summary = cbind(paste0(ngenes.present,"/",ngenes.present+ngenes.missing),c("PASS","FAIL")[signatures.failed+1],missing.genenames)
    dimnames(sig.qc.summary) = list(signatures.all,c("Genes present","Sufficient genes","Signature genes missing from codeset"))
    write.csv(sig.qc.summary,paste0(path.to.pathscore.results,"//codeset coverage of signature genes.csv"))
  }


#  #### QC plots for only the PC1 method:  
#  if(path.scoring.method=="PC1")
#  {
#    ## what QC plots should we show? percent of var? weighting barplots? ##
#  }
#  
#  #### plot the signature results, whatever the method:
#  signatures = colnames(sigs)
#  ## for each signature: score vs. PCA of its genes:
#  for(i in 1:length(signatures))
#  {
#    # get PCs of the data:
#    tempgenes = rownames(signaturematrix.final)[signaturematrix.final[,signatures[i]]!=0]
#    tempdat = data[,tempgenes]
#    print(dim(tempdat))
#
#    pc = prcomp(tempdat,scale=TRUE)
#    ## plot the sig score vs the top 3 pcs
#    # draw pairs plots of algorithm scores and PCs:
#    for(j in 1:length(pairscols))
#    {
#      for(r in 1:length(plottypearg))
#      {
#        plottype=plottypearg[r];
#      
#        if(length(pairscols)>1){tempfilename = drawplot(paste(path.to.pathscore.results,"//","principal components of signature genes - ",signatures[i]," - pairs plot colored by ",covariate.names.for.pairsplots[j],sep=""),plottype)}
#        if(length(pairscols)==1){tempfilename = drawplot(paste(path.to.pathscore.results,"//","principal components of signature genes - ",signatures[i]," - pairs plot",sep=""),plottype)}
#        tempfilename=gsub(path.results,"results",tempfilename)
#        pch = 1+15*(nrow(pc$x)<100)
#        pairs(cbind(sigs[,i],pc$x[,1:min(3,dim(pc$x)[2])]),labels=c(colnames(sigs)[i],paste0("PC",1:min(3,dim(pc$x)[2]))),col=pairscols[[j]],xaxt="n",yaxt="n",pch=pch)
#        dev.off()
#      }  
#    }
#    
#    ## plot the sig score wts and the loadings of the PCs:
#    for(r in 1:length(plottypearg)){
#      plottype=plottypearg[r];
#      tempfilename = drawplot(filename=paste(path.to.pathscore.results,"//principal components of signature genes loadings - ",colnames(sigs)[i],sep=""),plottype,height=2)
#      tempfilename=gsub(path.results,"results",tempfilename)
#      showpcs = min(3,ncol(pc$rotation))
#      #par(mfrow = c(showpcs,1))        
#      layout(matrix(c(1:(showpcs+1)),4),heights = c(rep(1,showpcs-1),1.2)) 
#      par(mar = c(2,4,0,0))
#      barplot(signaturematrix.final[rownames(pc$rotation),signatures[i]],las=2,ylab=paste0(colnames(sigs)[i]," weights"),names.arg="")
#      for(j in 1:showpcs)
#      {
#        if(j<showpcs){barplot(pc$rotation[,j],las=2,ylab=paste0("PC",j," loading"),names.arg="")}
#        if(j==showpcs){par(mar = c(6,4,0,0));barplot(pc$rotation[,j],las=2,ylab=paste0("PC",j," loading"))}
#      }
#      dev.off()
#    }
#  }
 
   
  #### summary tab: heatmaps, PCA:
  ## if only one signature: 
  if(ncol(sigs)==1)
  {
    # histogram of scores:
    for(r in 1:length(plottypearg)){
      plottype=plottypearg[r];
      tempfilename = drawplot(filename=paste(path.to.pathscore.results,"//heatmap of signatures",sep=""),plottype,width=1.5,height=1.5)
      tempfilename=gsub(path.results,"results",tempfilename)
      par(mar=c(10,4,2,1))
      hist(sigs[,1],xlab = colnames(sigs)[1],main="")
      dev.off()}
  }
  
  ## if >1 signature:
  if(ncol(sigs)>1)
  {
    # heatmap of signatures
    for(r in 1:length(plottypearg)){
      plottype=plottypearg[r];
      tempfilename = drawplot(filename=paste(path.to.pathscore.results,"//heatmap of signatures",sep=""),plottype,width=1.5,height=1.5)
      tempfilename=gsub(path.results,"results",tempfilename)
      par(oma=c(1,2,1,1))  
      par(mar=c(10,4,2,7))       
      #par(mar=c(10,4,2,1))
      hmcols<-colorRampPalette(c("cornflowerblue","black","orange"))(256)
      if(length(covariates)>0){ heatmap.plus.modified(sigs,scale="column",symm=FALSE,Colv=TRUE,RowSideColors=annotcols[,colnames(covariates),drop=FALSE],margins=c(9,11),cexCol = 0.85, cexRow = 0.75,col=hmcols, labRow = strtrim(abbreviate(toupper(rownames(sigs))), width = 30)) }   #$#  specified cexCol
      if(length(covariates)==0){ heatmap.plus.modified(sigs,scale="column",symm=FALSE,Colv=TRUE,margins=c(9,11),cexCol = 0.85, cexRow = 0.75,col=hmcols, labRow = strtrim(abbreviate(toupper(rownames(sigs))), width = 30)) }   #$#  specified cexCol
    dev.off()}
        
    # heatmap of cormat:
    for(r in 1:length(plottypearg)){
      plottype=plottypearg[r];
      tempfilename = drawplot(filename=paste(path.to.pathscore.results,"//signatures correlation heatmap",sep=""),plottype,width=1.7,height=1.7,heatmapres=TRUE)
      tempfilename=gsub(path.results,"results",tempfilename)
      par(mar=c(10,4,2,1))
      hmcols<-colorRampPalette(c("blue","antiquewhite3","firebrick"))(256)
      hmcols = hmcols
      cormat2 = cor(sigs)
      breaks = seq(-max(abs(cormat2)),max(abs(cormat2)),length.out=length(hmcols)+1)
      heatmap.2(cormat2,col=hmcols,symm=TRUE,Colv="Rowv",margins=c(8,8),cexRow=0.75,cexCol=0.75,trace="n",density="none", keysize = 1, key.xlab = "correlation")
      dev.off()}   
  
    # PCA of signatures:
    pc = prcomp(sigs,scale=TRUE) 
    # draw pairs plots:
    for(i in 1:length(pairscols))
    {
      for(r in 1:length(plottypearg))
      {
        plottype=plottypearg[r];
        
        if(length(pairscols)>1){tempfilename = drawplot(paste(path.to.pathscore.results,"//","principal components fit to signature scores - pairs plot colored by ",covariate.names.for.pairsplots[i],sep=""),
                                                        plottype)}
        if(length(pairscols)==1){tempfilename = drawplot(paste(path.to.pathscore.results,"//","principal components fit to signature scores pairs plot",sep=""),
                                                         plottype)}
        tempfilename=gsub(path.results,"results",tempfilename)
        pch = 1+15*(nrow(pc$x)<100)
        pairs(pc$x[,1:min(4,dim(pc$x)[2])],col=pairscols[[i]],xaxt="n",yaxt="n",pch=pch)
        #write.csv(pairscols[[i]],file=paste(path.to.csvs,"//pathway scores - PC pairs plot color",i,".csv",sep=""))   
      dev.off()}  
    }
    
    # and draw the PC loadings:  #$#
    for(r in 1:length(plottypearg)){
      plottype=plottypearg[r];
      tempfilename = drawplot(filename=paste(path.to.pathscore.results,"//principal components of signatures loadings",sep=""),plottype,height=2)
      tempfilename=gsub(path.results,"results",tempfilename)
      showpcs = min(4,ncol(pc$rotation))
      #par(mfrow = c(showpcs,1))        
      layout(matrix(c(1:showpcs),4),heights = c(rep(1,showpcs-1),1.2)) 
      par(mar = c(2,4,0,0))
      for(j in 1:showpcs)
      {
        if(j<showpcs){barplot(pc$rotation[,j],las=2,ylab=paste0("PC",j," loading"),names.arg="")}
        if(j==showpcs){par(mar = c(6,4,0,0));barplot(pc$rotation[,j],las=2,ylab=paste0("PC",j," loading"))}
      }
    dev.off()}
    
    
#      ## pairs plots of signatures:
#      nsignatures = dim(sigs)[2]
#      # assign colors if there are any covariates selected to plot against:
#      pairscols = list(); pairscols[[1]]="black"
#      covariate.names.for.pairsplots = c("")
#      #if(plot.against.covariates&(dim(covariates)[2]>0))
#      if(plot.against.covariates&(length(covariates)>0))
#      {
#        for(i in 1:dim(covariates)[2])
#        {
#          pairscols[[i]] = annotcols[,i]
#          covariate.names.for.pairsplots[i] = dimnames(covariates)[[2]][i]
#        }
#      }
    # draw pairs plots:
    for(i in 1:length(pairscols))
    {
      for(k in 1:dim(sigs)[2])
      {
        # unadjusted scores plot:
        for(r in 1:length(plottypearg)){
          plottype=plottypearg[r];
          signaturepairsplotsize = 1
          if(dim(sigs)[2]>4){signaturepairsplotsize = 1.5}
          if(dim(sigs)[2]>=16){signaturepairsplotsize = 2}
          if(nchar(covariate.names.for.pairsplots[1])>0){tempfilename = drawplot(paste(path.to.pathscore.results,"//","signatures pairs plot - ",dimnames(sigs)[[2]][k]," - colored by ",covariate.names.for.pairsplots[i],sep=""),
                                                                                 plottype,width = signaturepairsplotsize,height=signaturepairsplotsize)}
          if(nchar(covariate.names.for.pairsplots[1])==0){tempfilename = drawplot(paste(path.to.pathscore.results,"//","signatures pairs plot - ",dimnames(sigs)[[2]][k],sep=""),
                                                                                  plottype,width = signaturepairsplotsize,height=signaturepairsplotsize)}
          tempfilename=gsub(path.results,"results",tempfilename)
          
          par(mfrow=c(ceiling(sqrt(dim(sigs)[2])),ceiling(sqrt(dim(sigs)[2])))) 
          par(mar = c(4,6,3,0))
          for(j in setdiff(1:dim(sigs)[2],k))
          {
            plot(sigs[,k],sigs[,j],xlab=dimnames(sigs)[[2]][k],ylab=dimnames(sigs)[[2]][j],
                 col=pairscols[[i]],cex.lab=1.2,pch=16)
          }
        dev.off()}     
      }
    }  
  }# end if(dim(sigs)[2]>1
    ### compare to covariates
      

  
  
  ## plot scores by covariates and save univariate pvals:
  pvals=NULL
  #nsignatures = dim(sigs)[2]
    
  # loop through pathways:  
  if(plot.against.covariates&(length(covariates)>0))
  {
    # matrix of pvalues vs covariates:
    #pvals = matrix(NA,nsignatures,dim(covariates)[2])
    pvals = matrix(NA,ncol(sigs),dim(covariates)[2])
    #dimnames(pvals)[[1]] = dimnames(sigs)[[2]]
    #dimnames(pvals)[[2]] = dimnames(covariates)[[2]]
    dimnames(pvals) = list(dimnames(sigs)[[2]],dimnames(covariates)[[2]])
    for(j in 1:ncol(sigs))
    {
      #loop through covariates
      for(i in 1:dim(covariates)[2])
      {
        # ID type of covariate
        if(covariate.type[which(names(covariate.type) == names(covariates)[i])] == "continuous")
        {
          for(r in 1:length(plottypearg)){
            plottype=plottypearg[r];
            #tempfilename = drawplot(paste(path.to.pathscore.results,"//",names(covariates)[i]," - ",dimnames(sigs)[[2]][j],sep=""),plottype)
            tempfilename = drawplot(paste(path.to.pathscore.results,"//",dimnames(sigs)[[2]][j]," pathway scores vs. ",names(covariates)[i],sep=""),plottype)
            tempfilename=gsub(path.results,"results",tempfilename)
            #if(r==1)
            #{  	 
            #  strTemp=paste("document.write('              	    		  	   <li> Score vs ",names(covariates)[i],"<br><img src=\"",tempfilename,".png\"></li>');\n",sep="")
            #  cat(strTemp,file=paste(path.inc,"//panel2_1.js",sep=""),append=TRUE)
            #}
            plot(sigs[,j]~covariates[,i],xlab = dimnames(covariates)[[2]][i],ylab=paste(dimnames(sigs)[[2]][j],"score"),cex.lab=1.5)
            notmissing = !is.na(covariates[,i])
            lines(lowess(covariates[notmissing,i],sigs[notmissing,j]),col="grey25")
            dev.off()}
      # and run hypothesis test:
          pvals[j,i] = summary(lm(sigs[,j]~covariates[,i]))$coefficients[2,4]
        }
        if(covariate.type[which(names(covariate.type) == names(covariates)[i])] == "categorical")
        {
          x = as.factor(covariates[,i])
          y = as.vector(sigs[,j])
          for(r in 1:length(plottypearg)){
            plottype=plottypearg[r];
            #tempfilename = drawplot(paste(path.to.pathscore.results,"//covariate vs score - ",colnames(covariates)[i]," - ",dimnames(sigs)[[2]][j],sep=""),plottype)
            tempfilename = drawplot(paste(path.to.pathscore.results,"//",dimnames(sigs)[[2]][j]," pathway scores vs. ",names(covariates)[i],sep=""),plottype)
            tempfilename=gsub(path.results,"results",tempfilename)
            #if(r==1)
            #{  
            #  strTemp=paste("document.write('              	    		  	   <li> Score vs ",names(covariates)[i],"<br><img src=\"",tempfilename,".png\"></li>');\n",sep="")
            #  cat(strTemp,file=paste(path.inc,"//panel2_1.js",sep=""),append=TRUE)
            #}
            par(mar=c(10,6,2,1))
            bp=boxplot(y~x,border="darkgrey",ylab=paste(dimnames(sigs)[[2]][j],"score"),outline=FALSE,las=2,cex.lab=1.5)
            points(y~jitter(as.numeric(x)),col=annotcols2[[i]][x],pch=16)
            mtext(names(covariates)[i], side = 1, line = 6, font = 1, cex = 1.5)
            #lines(1:length(unique(x)),by(y,x,mean))
            dev.off()}
          # and run hypothesis test:
          complete = !is.na(covariates[,i])
          pvals[j,i] = anova(lm(sigs[complete,j]~covariates[complete,i]),lm(sigs[complete,j]~1))[[6]][2]
        }
        ### trend plots vs covariates:
        if(j==1)
        {
          for(r in 1:length(plottypearg)){
            plottype=plottypearg[r];
            tempfilename = drawplot(paste(path.to.pathscore.results,"//","trend plot of signatures vs. ",covariate.names.for.pairsplots[i],sep=""),
                                    plottype,width=2)
            trendplot(sigs,covariate=covariates[,i],covariate.name=dimnames(covariates)[[2]][i],
                      covariate.type=covariate.type[which(names(covariate.type) == names(covariates)[i])],center=FALSE,scale=FALSE,ylab="signatures",lwd=2,lty = rep(1:6,100)[1:ncol(sigs)])
          dev.off()} 
        }
      }
    }
  }
  # write .csvs of signatures:
  write.csv(sigs,file=paste(path.to.pathscore.results,"//signature scores.csv",sep=""))
}
